import ccxt
import json
import time
import os
import requests
import pandas as pd
import csv
from ta.momentum import RSIIndicator
from ta.trend import EMAIndicator
from dotenv import load_dotenv
from datetime import datetime

load_dotenv()

# Configuración
API_KEY = os.getenv("BINANCE_API_KEY")
API_SECRET = os.getenv("BINANCE_API_SECRET")
TELEGRAM_TOKEN = os.getenv("TELEGRAM_BOT_TOKEN")
CHAT_ID = os.getenv("TELEGRAM_CHAT_ID")
SYMBOLS = ['PEPE/USDT', 'MEME/FDUSD']
STOP_LOSS_PERCENT_MEME = 1
STOP_LOSS_PERCENT_BTC = 0.75
STOP_LOSS_PERCENT = 75
PROFIT_PERCENT_MEME = 2
PROFIT_PERCENT_BTC = 1.5
TIMEFRAME = '1m'
TOPE_COMPRA_MEME = 0.995
TOPE_COMPRA_BTC = 0.99
RSI_LIMIT_BTC = 60
RSI_LIMIT_MEME = 60

binance = ccxt.binance({
    'apiKey': API_KEY,
    'secret': API_SECRET,
    'enableRateLimit': True,
    'options': {'defaultType': 'spot'}
})

BASE_DIR = os.path.dirname(os.path.abspath(__file__))
# 📄 Archivo donde se guardarán las operaciones
LOG_FILE = os.path.join(BASE_DIR, "trades_log.csv")

def get_state_file(symbol):
    return os.path.join(BASE_DIR, f'state_{symbol.replace("/", "_").lower()}.json')

def load_state(symbol):
    file = get_state_file(symbol)
    if os.path.exists(file):
        with open(file) as f:
            return json.load(f)
    return {"position": False, "buy_price": 0.0}

def save_state(symbol, state):
    with open(get_state_file(symbol), 'w') as f:
        json.dump(state, f)

def send_telegram(msg):
    url = f"https://api.telegram.org/bot{TELEGRAM_TOKEN}/sendMessage"
    payload = {"chat_id": CHAT_ID, "text": msg}
    requests.post(url, data=payload)

def fetch_ohlcv(symbol, timeframe=TIMEFRAME, limit=120):
    return binance.fetch_ohlcv(symbol, timeframe, limit=limit)

def calculate_indicators(ohlcv):
    closes = [candle[4] for candle in ohlcv]
    rsi = RSIIndicator(pd.Series(closes)).rsi().iloc[-1]
    ema = EMAIndicator(pd.Series(closes), window=12).ema_indicator().iloc[-1]
    return rsi, ema, closes[-1]

def place_order(symbol, side, amount):
    return binance.create_market_order(symbol, side, amount)

def get_token_amount(symbol):
    token = symbol.split('/')[0]
    balance = binance.fetch_balance()
    return balance['total'].get(token, 0)

def get_base_balance(symbol):
    base = symbol.split('/')[1]
    balance = binance.fetch_balance()
    return balance['total'].get(base, 0)
    
def calculate_ema(ohlcv, period):
    closes = [candle[4] for candle in ohlcv]
    df = pd.DataFrame({'close': closes})
    return df['close'].ewm(span=period, adjust=False).mean().iloc[-1]

def process_symbol(symbol):
    state = load_state(symbol)
    ohlcv = fetch_ohlcv(symbol)
    rsi, ema12, current_price = calculate_indicators(ohlcv)
    base_balance = get_base_balance(symbol)
    buy_price = state["buy_price"]
    
    # Calcular el máximo de los últimos 2 días (48 velas si son de 1h)
    high_prices = [candle[2] for candle in ohlcv[-120:]]
    recent_high = max(high_prices)

    markets = binance.load_markets()
    market = markets[symbol]
    min_amount = market['limits']['amount']['min']
    
    # Ajuste especial si es BTC
    if "BTC" in symbol:
        profit_percent_used = PROFIT_PERCENT_BTC
        rsi_limit = RSI_LIMIT_BTC
        price_threshold = recent_high * TOPE_COMPRA_BTC
        stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_BTC / 100)
    else:
        profit_percent_used = PROFIT_PERCENT_MEME
        rsi_limit = RSI_LIMIT_MEME
        price_threshold = recent_high * TOPE_COMPRA_MEME
        stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_MEME / 100)

    if not state["position"]:
        # Entrada más agresiva
        if rsi <= rsi_limit and current_price >= ema12 and current_price <= price_threshold:
            amount_to_spend = base_balance * 1  # Usar 100% del capital disponible
            ticker = binance.fetch_ticker(symbol)
            raw_amount = amount_to_spend / ticker['ask']
            amount = float(binance.amount_to_precision(symbol, raw_amount))

            if amount >= min_amount:
                order = place_order(symbol, 'buy', amount)
                state["position"] = True
                state["buy_price"] = order['average']
                profit_price = order['average'] * (1 + profit_percent_used / 100)
                save_state(symbol, state)
                send_telegram(f"✅ Compra rápida: {amount:.6f} {symbol} a {order['average']:.8f} / Precio venta: {profit_price:.8f}")
                
                # Registrar la operación en el CSV
                price = order['average']
                total_usd = price * amount
                fee = order['fees'][0]['cost'] if order.get('fees') and len(order['fees']) > 0 else 0.0

                log_trade("BUY", symbol, amount, price, total_usd, fee)
 
    else:
        #token_amount = get_token_amount(symbol)
        current_price = binance.fetch_ticker(symbol)['last']
        #stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT / 100)
        take_profit_price = buy_price * (1 + profit_percent_used / 100)
        
        if "BTC" in symbol:
            token_amount = get_token_amount(symbol)
            if token_amount <= 0.00001:
                token_amount = 0.0
        else:
            token_amount = get_token_amount(symbol)
            if token_amount <= 0.99:
                token_amount = 0.0

        # Venta rápida por SL o TP
        if token_amount > 0 and state["position"]:
            if current_price <= stop_loss_price or current_price >= take_profit_price:
                order = place_order(symbol, 'sell', token_amount)
                sell_price = order['average']
                profit_percent = ((sell_price - buy_price) / buy_price) * 100
                reason = "🔻 SL rápido" if current_price <= stop_loss_price else "📈 TP rápido"

                send_telegram(
                    f"{reason}\n"
                    f"💰 Vendido: {token_amount:.6f} {symbol} a {sell_price:.8f}\n"
                    f"📉 Compra: {buy_price:.8f} | 📊 Ganancia: {profit_percent:.2f}%"
                )

                # Reiniciar estado para entrar de nuevo rápido
                state["position"] = False
                state["buy_price"] = 0.0
                save_state(symbol, state)
                
                # Registrar la operación en el CSV
                sell_price = order['average']
                total_usd = sell_price * token_amount
                fee = order['fees'][0]['cost'] if order.get('fees') and len(order['fees']) > 0 else 0.0
                profit_loss = (sell_price - buy_price) * token_amount

                log_trade("SELL", symbol, token_amount, sell_price, total_usd, fee, profit_loss)


# 📝 Función para registrar una operación
def log_trade(action, symbol, amount, price, total_usd, fee=0.0, profit_loss=0.0):
    """
    action: 'BUY' o 'SELL'
    symbol: par de trading
    amount: cantidad comprada o vendida
    price: precio unitario
    total_usd: total de la operación en USD
    fee: comisión pagada
    profit_loss: ganancia o pérdida (solo para ventas)
    """
    # Crear archivo con encabezados si no existe
    file_exists = os.path.isfile(LOG_FILE)
    with open(LOG_FILE, mode="a", newline="") as file:
        writer = csv.writer(file)
        if not file_exists:
            writer.writerow(["Fecha", "Accion", "Symbol", "Cantidad", "Precio", "Total USD", "Comision", "Ganancia/Perdida"])
        writer.writerow([
            datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            action,
            symbol,
            amount,
            price,
            total_usd,
            fee,
            profit_loss
        ])

    # 📢 Notificación por Telegram
    msg = f"📊 {action} {symbol}\n" \
          f"📦 Cantidad: {amount:.8f}\n" \
          f"💰 Precio: {price:.8f}\n" \
          f"💵 Total USD: {total_usd:.8f}\n" \
          f"💸 Comisión: {fee:.8f}\n"
    if action == "SELL":
        total_neto = total_usd - fee
        msg += f"📈 Ganancia/Pérdida: {profit_loss:.8f} USD\n" \
               f"💰 Total neto: {total_neto:.8f} USD"
    send_telegram(msg)

def main():
    send_telegram("🤖 Bot multipar iniciado")
    last_report_time = 0

    while True:
        try:
            for symbol in SYMBOLS:
                
                process_symbol(symbol)

            # Reporte horario
            current_time = time.time()
            if current_time - last_report_time >= 3600:
                report = "🕒 Reporte horario:\n"
            
                for symbol in SYMBOLS:
                    try:
                        state = load_state(symbol)
                        rsi, ema, current_price = calculate_indicators(fetch_ohlcv(symbol))
                        buy_price = state.get('buy_price', 0.0)
                        in_position = state.get('position', False)
                        stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT / 100)
                        
                        # Máximo de los últimos 7 días (opcional)
                        ohlcv = fetch_ohlcv(symbol, '1m', limit=120)
                        highest_high = max(candle[2] for candle in ohlcv)
                        
                        if "BTC" in symbol:
                            profit_percent_used = PROFIT_PERCENT_BTC  # profit más bajo para scalping BTC
                            threshold_price = highest_high * TOPE_COMPRA_BTC
                            stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_BTC / 100)
                        else:
                            profit_percent_used = PROFIT_PERCENT_MEME
                            threshold_price = highest_high * TOPE_COMPRA_MEME
                            stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_MEME / 100)
                            
                        profit_target = buy_price * (1 + profit_percent_used/ 100)                        
            
                        if "BTC" in symbol:
                            report += (
                                f"\n📈 {symbol}\n"
                                f"💰 RSI: {rsi:.2f} | EMA12: {ema:.2f} | Precio actual: {current_price:.2f}\n"
                                f"💰 Compra: {buy_price:.2f} | StopLoss: {stop_loss_price:.2f}\n"
                                f"💹 Profit target: {profit_target:.2f}\n"
                                f"📊 En posición: {in_position}\n"
                                f"📉 Tope semanal: {highest_high:.2f} | 99.5% del tope: {threshold_price:.2f}\n"
                            )
                        else:
                            report += (
                                f"\n📈 {symbol}\n"
                                f"💰 RSI: {rsi:.2f} | EMA12: {ema:.8f} | Precio actual: {current_price:.8f}\n"
                                f"💰 Compra: {buy_price:.8f} | StopLoss: {stop_loss_price:.8f}\n"
                                f"💹 Profit target: {profit_target:.8f}\n"
                                f"📊 En posición: {in_position}\n"
                                f"📉 Tope semanal: {highest_high:.8f} | 99.5% del tope: {threshold_price:.8f}\n"
                            )
            
                    except Exception as e:
                        report += f"\n⚠️ Error con {symbol}: {str(e)}\n"
            
                # Balance general
                try:
                    balance_data = binance.fetch_balance()
                    report += f"\n💼 Balance general:\n"
                
                    for coin in ['USDT', 'USDC', 'BONK', 'PEPE', 'HAEDAL', 'EUR', 'SHIB','MEME', 'FDUSD', 'HMSTR', 'BTC']:
                        amount = balance_data['total'].get(coin, 0)
                        if amount > 0:
                            # Determinar contra qué moneda cotizar (quote)
                            if coin in ['BONK', 'HAEDAL', 'BTC']:
                                quote = 'USDC'
                            elif coin in ['PEPE', 'EUR', 'HMSTR', 'SHIB']:
                                quote = 'USDT'
                            elif coin == 'MEME':
                                quote = 'FDUSD'
                            else:
                                quote = None  # Ya son monedas base como USDT o USDC
                
                            if quote:
                                try:
                                    price = binance.fetch_ticker(f'{coin}/{quote}')['last']
                                    estimated = amount * price
                                    if coin == 'BTC':
                                        report += f" - {coin}: {amount:.8f} ≈ {estimated:.4f} {quote}\n"
                                    else:
                                        report += f" - {coin}: {amount:.4f} ≈ {estimated:.4f} {quote}\n"
                                except:
                                    report += f" - {coin}: {amount:.4f} (sin cotización)\n"
                            else:
                                report += f" - {coin}: {amount:.4f}\n"
                
                except Exception as e:
                    report += f"\n⚠️ Error al obtener balances: {str(e)}\n"
            
                send_telegram(report)
                last_report_time = current_time

            time.sleep(3)

        except Exception as e:
            print(f"Error: {str(e)}")
            send_telegram(f"⚠️ Error: {str(e)}")
            time.sleep(60)

if __name__ == "__main__":
    main()
